import pygame, sys, argparse, random, json, os, traceback, time as time_mod
from dataclasses import dataclass

# --------- Config ----------
WIDTH, HEIGHT = 1920, 1080
FPS = 60

LANES = 3
LANE_PADDING = 200

ROAD = (40, 40, 48)
LANE_MARK = (200,200,200)
PLAYER_DEFAULT = (62,150,255)
COIN_COLOR = (255,200,50)
TRAFFIC_COLOR = (220,70,70)
TEXT_COLOR = (235,235,235)

SPAWN_INTERVAL = 0.9
SPEED_BASE = 520.0
PLAYER_SWITCH_TIME = 0.12

LAST_RUN_FN = "last_run_result.json"
CRASH_LOG_FN = "crash_log.txt"

# --------- Data ----------
@dataclass
class Traffic:
    lane: int
    y: float
    w: int
    h: int
    speed: float
    def rect(self, lane_x):
        return pygame.Rect(int(lane_x[self.lane] - self.w//2),
                           int(self.y - self.h//2),
                           self.w, self.h)

@dataclass
class Coin:
    lane: int
    y: float
    r: int = 12
    def rect(self, lane_x):
        return pygame.Rect(int(lane_x[self.lane] - self.r),
                           int(self.y - self.r),
                           self.r*2, self.r*2)

# --------- Helpers ----------
def hex_to_rgb(hexstr):
    try:
        h = hexstr.lstrip("#")
        return tuple(int(h[i:i+2],16) for i in (0,2,4))
    except:
        return PLAYER_DEFAULT

def draw_text(screen, txt, size, color, x, y):
    font = pygame.font.SysFont("Arial", size)
    screen.blit(font.render(txt, True, color), (x,y))

# --------- Args ----------
def parse_args():
    p = argparse.ArgumentParser()
    p.add_argument("--name", default="Player")
    p.add_argument("--color", default="#3E96FF")
    p.add_argument("--fullscreen", default="0")
    p.add_argument("--res", default=f"{WIDTH}x{HEIGHT}")
    return p.parse_args()

# --------- Save Result ----------
def write_result_and_quit(score, args, distance=0, coins=0, time_elapsed=0, success=False):
    try:
        with open(LAST_RUN_FN, "w") as f:
            json.dump({
                "score": score,
                "player": args.name,
                "distance": int(distance),
                "coins": coins,
                "time_elapsed": time_elapsed,
                "success": success
            }, f)
    except:
        pass
    pygame.quit()
    sys.exit()

# --------- Main ----------
def main():
    args = parse_args()
    pygame.init()
    pygame.font.init()

    w, h = map(int, args.res.split("x"))
    flags = pygame.FULLSCREEN if args.fullscreen == "1" else 0
    screen = pygame.display.set_mode((w, h), flags)
    pygame.display.set_caption("Endless Highway — " + args.name)
    clock = pygame.time.Clock()

    # lanes
    lane_x = []
    lane_w = (screen.get_width() - LANE_PADDING*2) / LANES
    for i in range(LANES):
        lane_x.append(int(LANE_PADDING + lane_w/2 + i*lane_w))

    # player
    player_lane = 1
    player_x = lane_x[player_lane]
    player_y = int(screen.get_height() * 0.78)
    player_w, player_h = 72, 120

    traffic, coins = [], []
    spawn_timer = 0
    distance = 0
    coins_collected = 0
    start_time = time_mod.time()
    elapsed_total = 0

    running = True
    while running:
        dt = clock.tick(FPS)/1000
        elapsed_total += dt

        difficulty = min(1.0 + elapsed_total * 0.035, 6.0)

        # events
        for ev in pygame.event.get():
            if ev.type == pygame.QUIT:
                write_result_and_quit(0, args, distance, coins_collected, time_mod.time()-start_time)
            if ev.type == pygame.KEYDOWN:
                if ev.key == pygame.K_ESCAPE:
                    write_result_and_quit(0, args, distance, coins_collected, time_mod.time()-start_time)
                if ev.key in (pygame.K_LEFT, pygame.K_a):
                    player_lane = max(0, player_lane - 1)
                if ev.key in (pygame.K_RIGHT, pygame.K_d):
                    player_lane = min(LANES-1, player_lane + 1)

        # smooth movement
        player_x += (lane_x[player_lane] - player_x) * min(1.0, dt * (1/PLAYER_SWITCH_TIME))

        # spawn
        spawn_timer -= dt * difficulty
        if spawn_timer <= 0:
            spawn_timer = SPAWN_INTERVAL / difficulty
            lane = random.randrange(LANES)
            ty = -150
            ts = SPEED_BASE * difficulty * (1 + random.random()*0.3)
            tw, th = random.randint(60,100), random.randint(120,160)
            traffic.append(Traffic(lane, ty, tw, th, ts))

            if random.random() < 0.4:
                coins.append(Coin(lane, -80))

        # update
        for t in list(traffic):
            t.y += t.speed * dt
            if t.y > screen.get_height()+200:
                traffic.remove(t)

        for c in list(coins):
            c.y += SPEED_BASE * dt * difficulty
            if c.y > screen.get_height()+100:
                coins.remove(c)

        player_rect = pygame.Rect(player_x-player_w//2, player_y-player_h//2, player_w, player_h)

        for t in traffic:
            if player_rect.colliderect(t.rect(lane_x)):
                write_result_and_quit(int(distance//10), args, distance, coins_collected, time_mod.time()-start_time)

        for c in list(coins):
            if player_rect.colliderect(c.rect(lane_x)):
                coins_collected += 1
                coins.remove(c)

        distance += SPEED_BASE * dt * difficulty
        score = int(distance//10) + coins_collected * 50

        # draw
        screen.fill((20,120,60))
        road = pygame.Rect(LANE_PADDING, h*0.15, w-LANE_PADDING*2, h*0.78)
        pygame.draw.rect(screen, ROAD, road, border_radius=8)

        for i in range(1, LANES):
            x = LANE_PADDING + i*((w-LANE_PADDING*2)/LANES)
            y = road.top+20
            while y < road.bottom-20:
                pygame.draw.line(screen, LANE_MARK, (x,y), (x,y+12), 6)
                y += 30

        for t in traffic:
            pygame.draw.rect(screen, TRAFFIC_COLOR, t.rect(lane_x), border_radius=6)

        for c in coins:
            pygame.draw.circle(screen, COIN_COLOR, (lane_x[c.lane], int(c.y)), c.r)

        pygame.draw.rect(screen, hex_to_rgb(args.color), player_rect, border_radius=10)

        draw_text(screen, args.name, 26, TEXT_COLOR, 20, 20)
        draw_text(screen, f"Score: {score}", 26, TEXT_COLOR, 20, 55)
        draw_text(screen, f"Coins: {coins_collected}", 22, TEXT_COLOR, 20, 90)
        draw_text(screen, f"Speed: {difficulty:.2f}x", 20, TEXT_COLOR, 20, 125)
        draw_text(screen, f"Distance: {int(distance)}", 20, TEXT_COLOR, w-280, 20)

        pygame.display.flip()

# --------- Crash ----------
if __name__ == "__main__":
    try:
        main()
    except Exception:
        with open(CRASH_LOG_FN, "w") as f:
            f.write(traceback.format_exc())
        pygame.quit()
        sys.exit(1)
